import { NextRequest, NextResponse } from 'next/server';
import Database from 'better-sqlite3';
import path from 'path';

function getDb() {
    const dbPath = path.join(process.cwd(), 'domains.db');
    return new Database(dbPath);
}

export async function GET(_: NextRequest, { params }: { params: Promise<{ id: string }> }) {
    const { id } = await params;
    const db = getDb();

    try {
        // Get domains count for this customer
        const domainsResult = db.prepare(`
            SELECT 
                COUNT(*) as total_domains,
                COUNT(CASE WHEN status = 'paid' THEN 1 END) as active_domains,
                COUNT(CASE WHEN nextPayment <= date('now', '+30 days') AND nextPayment > date('now') THEN 1 END) as expiring_soon
            FROM domains 
            WHERE customer_id = ?
        `).get(id) as { total_domains: number; active_domains: number; expiring_soon: number };

        // Get tasks count for this customer
        const tasksResult = db.prepare(`
            SELECT 
                COUNT(*) as total_tasks,
                COUNT(CASE WHEN status IN ('pending', 'in_progress') THEN 1 END) as active_tasks,
                COUNT(CASE WHEN priority = 'urgent' AND status IN ('pending', 'in_progress') THEN 1 END) as urgent_tasks
            FROM tasks 
            WHERE customer_id = ?
        `).get(id) as { total_tasks: number; active_tasks: number; urgent_tasks: number };

        // Get projects count for this customer
        const projectsResult = db.prepare(`
            SELECT 
                COUNT(*) as total_projects,
                COUNT(CASE WHEN status = 'active' THEN 1 END) as active_projects,
                COUNT(CASE WHEN status = 'completed' THEN 1 END) as completed_projects
            FROM projects 
            WHERE customer_id = ?
        `).get(id) as { total_projects: number; active_projects: number; completed_projects: number };

        // Get revenue for this customer (from quotes or payments)
        const revenueResult = db.prepare(`
            SELECT 
                COALESCE(SUM(amount), 0) as total_revenue,
                COALESCE(SUM(CASE WHEN status = 'unpaid' THEN amount ELSE 0 END), 0) as outstanding_debt
            FROM quotes 
            WHERE customer_id = ? AND status IN ('won', 'paid', 'unpaid')
        `).get(id) as { total_revenue: number; outstanding_debt: number };

        // Get events count for this customer
        const eventsResult = db.prepare(`
            SELECT COUNT(*) as upcoming_events
            FROM events 
            WHERE customer_id = ? AND start_datetime >= date('now')
        `).get(id) as { upcoming_events: number };

        const stats = {
            domains_count: domainsResult.total_domains,
            domains_expiring_soon: domainsResult.expiring_soon,
            active_tasks: tasksResult.active_tasks,
            urgent_tasks: tasksResult.urgent_tasks,
            total_tasks: tasksResult.total_tasks,
            active_projects: projectsResult.active_projects,
            total_projects: projectsResult.total_projects,
            total_revenue: revenueResult.total_revenue,
            outstanding_debt: revenueResult.outstanding_debt,
            upcoming_events: eventsResult.upcoming_events
        };

        return NextResponse.json(stats);
    } catch (e: any) {
        console.error('Get customer stats error:', e);
        return NextResponse.json({ error: 'שגיאה בטעינת סטטיסטיקות לקוח: ' + e.message }, { status: 500 });
    } finally {
        db.close();
    }
}
